<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\OrganizationModel;

class OrganizationsController extends BaseController
{
    protected $organizationModel;

    public function __construct()
    {
        $this->organizationModel = new OrganizationModel();
    }

    public function index()
    {
        $search = $this->request->getGet('search') ?? '';
        $filter = [
            'tahun_mulai' => $this->request->getGet('tahun_mulai') ?? '',
            'tahun_selesai' => $this->request->getGet('tahun_selesai') ?? ''
        ];
        $sort = $this->request->getGet('sort') ?? 'urutan';
        $order = $this->request->getGet('order') ?? 'ASC';
        $page = (int)($this->request->getGet('page') ?? 1);

        $builder = $this->organizationModel->searchAndFilter($search, $filter, $sort, $order);
        $total = $builder->countAllResults(false);
        
        $organizations = $builder->get(10, ($page - 1) * 10)->getResultArray();

        $pager = \Config\Services::pager();
        $pager->store('organizations', $page, 10, $total);

        $data = [
            'title' => 'Kelola Organisasi',
            'organizations' => $organizations,
            'pager' => $pager,
            'search' => $search,
            'filter' => $filter,
            'sort' => $sort,
            'order' => $order
        ];

        return view('admin/organizations/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Organisasi'
        ];
        return view('admin/organizations/create', $data);
    }

    public function store()
    {
        $rules = [
            'nama_organisasi' => 'required|min_length[3]|max_length[200]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'nama_organisasi' => $this->request->getPost('nama_organisasi'),
            'jabatan' => $this->request->getPost('jabatan'),
            'tahun_mulai' => $this->request->getPost('tahun_mulai'),
            'tahun_selesai' => $this->request->getPost('tahun_selesai'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'urutan' => (int)($this->request->getPost('urutan') ?? 0)
        ];

        if ($this->organizationModel->insert($data)) {
            return redirect()->to('admin/organizations')->with('success', 'Organisasi berhasil ditambahkan!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal menambahkan organisasi!');
        }
    }

    public function edit($id)
    {
        $organization = $this->organizationModel->find($id);
        
        if (!$organization) {
            return redirect()->to('admin/organizations')->with('error', 'Organisasi tidak ditemukan!');
        }

        $data = [
            'title' => 'Edit Organisasi',
            'organization' => $organization
        ];
        return view('admin/organizations/edit', $data);
    }

    public function update($id)
    {
        $organization = $this->organizationModel->find($id);
        
        if (!$organization) {
            return redirect()->to('admin/organizations')->with('error', 'Organisasi tidak ditemukan!');
        }

        $rules = [
            'nama_organisasi' => 'required|min_length[3]|max_length[200]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'nama_organisasi' => $this->request->getPost('nama_organisasi'),
            'jabatan' => $this->request->getPost('jabatan'),
            'tahun_mulai' => $this->request->getPost('tahun_mulai'),
            'tahun_selesai' => $this->request->getPost('tahun_selesai'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'urutan' => (int)($this->request->getPost('urutan') ?? 0)
        ];

        if ($this->organizationModel->update($id, $data)) {
            return redirect()->to('admin/organizations')->with('success', 'Organisasi berhasil diupdate!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal mengupdate organisasi!');
        }
    }

    public function delete($id)
    {
        $organization = $this->organizationModel->find($id);
        
        if (!$organization) {
            return redirect()->to('admin/organizations')->with('error', 'Organisasi tidak ditemukan!');
        }

        if ($this->organizationModel->delete($id)) {
            return redirect()->to('admin/organizations')->with('success', 'Organisasi berhasil dihapus!');
        } else {
            return redirect()->to('admin/organizations')->with('error', 'Gagal menghapus organisasi!');
        }
    }
}


